/*****************************************************************************
 * main.c 
 *
 * Project: SCT Application Example for LPC4300
 *
 * Description:
 *   Implements an example of output usage for events
 *
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 *****************************************************************************/


/* For this application the following interface pins are used:
 * ----------------------------------------------------------------------------------
 * 1) NXP validation board:
 *
 * Signal    Port   Connector              Signal    Port   Connector       Description
 * -----------------------------------------------------------------------------------
 * CTIN_3    P7_3   LCD_UNI, pin 22 <----> GPIO4_5   P8_5   LCD_EXT, pin 1  IN_SIGNAL
 * CTOUT_0   P4_2   LCD_UNI, pin 9                                          CONDITION_1
 * CTOUT_1   P4_1   LCD_UNI, pin 6                                          BRANCH1_ACTIVE 
 * CTOUT_2   P4_4   LCD_UNI, pin 7                                          CONDITION_2 
 * CTOUT_3   P4_3   LCD_UNI, pin 8                                          BRANCH2_ACTIVE
 *
 * -----------------------------------------------------------------------------------
 *
 *
 * 2) NXP evaluation board v4.1:
 *
 * Signal    Port   Connector          Signal    Port   Connector       Description
 * -----------------------------------------------------------------------------------
 * CTIN_3    P7_3   X1, pin 22 <---->  GPIO4_5   P8_5   JP9, pin 6      IN_SIGNAL
 * CTOUT_0   P4_2   X1, pin 3                                           CONDITION_1
 * CTOUT_1   P4_1   X1, pin 4                                           BRANCH1_ACTIVE 
 * CTOUT_2   P4_4   X1, pin 5                                           CONDITION_2 
 * CTOUT_3   P4_3   X1, pin 6                                           BRANCH2_ACTIVE
 *            
 * -----------------------------------------------------------------------------------
 *
 *
 * 3) Hitex board:  
 *
 * Signal    Port   Connector           Signal    Port   Connector       Description
 * -----------------------------------------------------------------------------------
 * CTIN_3    P7_3   SV18, pin 16 <----> GPIO4_5   P8_5   X11, pin 11     IN_SIGNAL
 * CTOUT_0   P4_2   X11, pin 7                                           CONDITION_1
 * CTOUT_1   P4_1   JP20, pin 2                                          BRANCH1_ACTIVE 
 * CTOUT_2   P4_4   JP22, pin 2                                          CONDITION_2 
 * CTOUT_3   P4_3   X11, pin 6                                           BRANCH2_ACTIVE
 *          
 * -----------------------------------------------------------------------------------
 */


#include <stdio.h>

#ifdef LPC43XX
    #include "LPC43xx_TC2.h"
    #include "scu_LPC43xx.h"
    #include "emc_LPC43xx.h"
#endif

#ifdef LPC18XX
    #include "LPC18xx.h"
    #include "scu_LPC18xx.h"
    #include "emc_LPC18xx.h"
#endif

#include "sct_fsm.h"




/******************************************************************************
 * Initialize board type specific IO's
 *  - NXP validation board
 *  - NXP evaluation board v4.1
 *  - Hitex board version A2
 ******************************************************************************/
void IOInit(void)
{

#ifdef NXP_VALIDATION_BOARD 
    // Board (and chip) specific initializations - NXP validation board 
    #ifdef LPC43XX 
        scu_pinmux(9, 2, MD_PDN, FUNC2);    // GPIO4_14: LD11   
        scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        LPC_GPIO4->DIR |= (1<<14);          // GPIO4_11 = output
        LPC_GPIO4->DIR |= (1<<11);          // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
    
        scu_pinmux (2, 3, MD_PDN, FUNC0);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22 
    #endif

    #ifdef LPC18XX 
        scu_pinmux(9, 2, MD_PDN, FUNC0);    // GPIO4_14: LD11

        // on LPC1800 port pin PA_4 is not available    
        //scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        //LPC_GPIO4->DIR |= (1<<14);            // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
    
        LPC_GPIO4->DIR |= (1<<14);          // LD11 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
    
        scu_pinmux (2, 3, MD_PDN, FUNC3);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22 
    #endif
#endif


#ifdef NXP_EVAL_BOARD   
    // Board (and chip) specific initializations - NXP eval board v4.1
    #ifdef LPC43XX
        scu_pinmux(8, 0, MD_PLN, FUNC3);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC3);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC3);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (2, 5,    MD_PDN, FUNC1);      // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif

    #ifdef LPC18XX
        scu_pinmux(8, 0, MD_PLN, FUNC0);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC0);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC0);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        scu_pinmux (2, 5,    MD_PDN, FUNC1);    // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif
#endif

#ifdef HITEX_BOARD
    // Board (and chip) specific initializations - Hitex board
    #ifdef LPC43XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC2);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC2);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC3);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC3);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux(4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux(4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux(4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux(4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux(4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux(4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
        scu_pinmux(7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - X11, pin 34
        scu_pinmux(0xB, 0, MD_PLN, FUNC1); // PB_0:  SCTOUT_10 - X11, pin 27
        scu_pinmux(7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - X11, pin 30
        scu_pinmux(7, 5, MD_PLN, FUNC1);   // P7_5:  SCTOUT_12 - SV18, pin 2
        scu_pinmux(7, 0, MD_PLN, FUNC1);   // P7_0:  SCTOUT_14 - JP19, pin 1
        scu_pinmux(7, 1, MD_PLN, FUNC1);   // P7_1:  SCTOUT_15 - X11, pin 22
    
        scu_pinmux(2, 4,  MD_PUP, FUNC0);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC0);  // P2_3:  SCTIN_1 - X3, pin 11
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13   // does not work
        //scu_pinmux(0xF, 8, MD_PUP, FUNC0);  // P4_10: SCTIN_2 - X5, pin 20   // does not work
        //scu_pinmux(2, 5,  MD_PUP, FUNC1);  // P2_5: SCTIN_2 - IC U13, pin 8   // does not work
        scu_pinmux(7, 3,  MD_PUP, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PUP, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11
    #endif
    
    #ifdef LPC18XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC0);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC0);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC0);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC0);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux(4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux(4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux(4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux(4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux(4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux(4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
    
        scu_pinmux(2, 4,  MD_PUP, FUNC3);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC3);  // P2_3:  SCTIN_1 - X3, pin 11
        scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13
        scu_pinmux(7, 3,  MD_PDN, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PDN, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11
    #endif
#endif
}



/******************************************************************************
 * Initialize special board specific IO
 * 
 *
 *  
 ******************************************************************************/
void SpecialIOInit(void)
{
    // Set another pin as GPIO function and use this GPIO to trigger the state machine on CTIN_3
    scu_pinmux (8, 5, MD_PDN, FUNC3);   // P8_5: GPIO 4_5
    
    LPC_GPIO4->DIR |= (1 << 5);         // output
    LPC_GPIO4->CLR = (1 << 5);          // set to zero
}



/******************************************************************************
 * SCT interrupt handler
 * 
 *
 *  
 ******************************************************************************/
void SCT_IRQHandler (void)
{
    // Acknowledge the interrupt source
    ;
}



/******************************************************************************
 * main 
 * 
 *
 *  
 ******************************************************************************/
int main (void) 
{
    uint32_t i;


    // Set the Vector table offset if the program code is in external memory
    #ifdef EXTFLASH
        extern unsigned long __Vectors;
        SCB->VTOR = (unsigned long)&__Vectors;
    #endif
    
    // System and board specific initializations 
    SystemInit();

    // Enable clocks and PLL
    LPC_CREG->CREG0 |= (1<<1)|(1<<0);               // Enable 32 kHz & 1 kHz on osc32k 
    SetPL160M(SRC_XTAL, 10);                    // Set PL160M @ 10*12 = 120 MHz with the XTAL as source clock 
    #ifdef LPC43XX
        SetClock(BASE_M4_CLK, SRC_PL160M_0, DIV1);      // Run base M3 clock from PL160M, div by 1 = no division 
    #endif
    #ifdef LPC18XX
        SetClock(BASE_M3_CLK, SRC_PL160M_0, DIV1);      // Run base M3 clock from PL160M, div by 1 = no division 
    #endif

    // Initialise the IO's needed for this example
    IOInit();

    // Configure the EMC in case the software runs from external flash 
    #ifdef EXTFLASH
        EMC_Config_Pinmux();
        EMC_Config_Static();
    #endif

    // Global configuration of the SCT 
    #ifdef LPC43XX
        LPC_CCU1->CLK_M4_SCT_CFG = 0x00000001;          // Enable SCT branch clock in CCU1, RUN=1, AUTO=0, WAKE=0 
    #endif
    #ifdef LPC18XX
        LPC_CCU1->CLK_M3_SCT_CFG = 0x00000001;          // Enable SCT branch clock in CCU1, RUN=1, AUTO=0, WAKE=0 
    #endif
    
    LPC_SCT->CTRL_L = 0x000C + ((100-1) << 5);          // L counter: set prescaler = 100. Clear counter. Keep halted
                                              
    //LPC_SCT->CTRL_H = 0x000C + ((100-1) << 5);        // H counter: set prescaler = 100. Clear counter. Keep halted

    // Now use the FSM code to configure the state machine 
    sct_fsm_init();

    // Other manual configuration and overrides 
    // Conflict resolution: CLR override for all
    LPC_SCT->RES = (2 << (2*0)) | (2 << (2*1)) | (2 << (2*2)) | (2 << (2*3));

    // Start the SCT
    LPC_SCT->CTRL_U &= ~(1 << 2);    // Run L only
    
    // Interrupts are not used in this example
    //NVIC_EnableIRQ (SCT_IRQn);

    // Run CPU in an endless loop
    while (1) 
    {
        // Generate the trigger signal which is connected to CTIN_3
        for(i = 0; i < 10000; i++); 
        LPC_GPIO4->SET = (1 << 5);          // create  rising edge

    
        for (i = 0; i < 10000; i++);
        LPC_GPIO4->CLR = (1 << 5);          // create falling edge
    }

}


